<?php

namespace App\Controller;

use App\Entity\Personne;
use App\Entity\User;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Core\Authentication\Token\Storage\TokenStorageInterface;
use Symfony\Component\Security\Core\Authentication\Token\UsernamePasswordToken;
use Symfony\Component\Security\Core\Exception\BadCredentialsException;
use Symfony\Component\Serializer\SerializerInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use OpenApi\Attributes as OA;

class ApiController extends AbstractController
{

    /**
     * List the rewards of the specified user.
     *
     * This call takes into account all confirmed awards, but not pending or refused awards.
     */
    #[Route('/api/personne', name: 'app_api', methods: ['GET'])]

    public function index(Request $request, ManagerRegistry $doctrine, SerializerInterface $serializer): JsonResponse
    {
        $entityManager = $doctrine->getManager();
        $queryBuilder = $entityManager->createQueryBuilder();

        $queryBuilder->select('p.id, p.name, p.firstName,p.image')
            ->addSelect('j.designation AS job_designation, pro.rs AS profile_rs, h.designation AS hobby_name')
            ->from(Personne::class, 'p')
            ->leftJoin('p.job', 'j')
            ->leftJoin('p.profile', 'pro')
            ->leftJoin('p.hobbies', 'h')
            ->orderBy('p.id', 'ASC');

        $query = $queryBuilder->getQuery();
        $results = $query->getResult();

        // Format the results to group hobbies by person
        $formattedData = [];
        foreach ($results as $result) {
            $personId = $result['id'];
            if (!isset($formattedData[$personId])) {
                $formattedData[$personId] = [
                    'id' => $result['id'],
                    'name' => $result['name'],
                    'firstName' => $result['firstName'],
                    'image' => $result['image'], //$request->getHost()."/uploads/personnes/".
                    'job_designation' => $result['job_designation'],
                    'profile_rs' => $result['profile_rs'],
                    'hobbies' => [],
                ];
            }
            // Add hobby only if it's not empty
            if ($result['hobby_name']) {
                $formattedData[$personId]['hobbies'][] = $result['hobby_name'];
            }
        }

        // Serialize the formatted data
        $data = $serializer->serialize(array_values($formattedData), 'json');

        return new JsonResponse($data, 200, [], true);
    }


    #[Route('/api/personne', name: 'api_personne_create', methods: ['POST'])]
    public function create(Request $request, ManagerRegistry $doctrine, ValidatorInterface $validator): JsonResponse
    {
        // Récupérer les données envoyées via la méthode POST
        $data = json_decode($request->getContent(), true);
        // Créer une nouvelle instance de l'entité Personne
        $personne = new Personne();
        $personne->setName($data['name']);
        $personne->setFirstName($data['firstName']);
        $personne->setAge($data['age']);
        // Ajouter d'autres champs selon votre entité Personne
        // Valider les données
        $errors = $validator->validate($personne);
        // S'il y a des erreurs de validation, retourner une réponse JSON avec les erreurs
        if (count($errors) > 0) {
            $errorMessages = [];
            foreach ($errors as $error) {
                $errorMessages[$error->getPropertyPath()] = $error->getMessage();
            }
            return new JsonResponse(['errors' => $errorMessages], JsonResponse::HTTP_BAD_REQUEST);
        }
        // Enregistrer la nouvelle personne dans la base de données
        $entityManager = $doctrine->getManager();
        $entityManager->persist($personne);
        $entityManager->flush();
        // Retourner une réponse JSON indiquant le succès de la création
        return new JsonResponse(['message' => 'Personne créée avec succès'], JsonResponse::HTTP_CREATED);
    }



    #[Route('/api/personne/{id}', name: 'api_personne_update', methods: ['PUT'])]
    public function update(Request $request, int $id, ManagerRegistry $doctrine): JsonResponse
    {
        // Récupérer les données envoyées via la méthode PUT
        $data = json_decode($request->getContent(), true);

        // Récupérer la personne à modifier depuis la base de données
        $entityManager = $doctrine->getManager();
        $personne = $entityManager->getRepository(Personne::class)->find($id);

        // Si la personne n'existe pas, retourner une erreur 404
        if (!$personne) {
            return new JsonResponse(['error' => 'Personne non trouvée'], JsonResponse::HTTP_NOT_FOUND);
        }

        // Mettre à jour les propriétés de la personne avec les nouvelles valeurs
        if (isset($data['name'])) {
            $personne->setName($data['name']);
        }
        if (isset($data['firstName'])) {
            $personne->setFirstName($data['firstName']);
        }
        if (isset($data['age'])) {
            $personne->setAge($data['age']);
        }
        // Ajouter d'autres champs selon votre entité Personne

        // Enregistrer les modifications dans la base de données
        $entityManager->flush();


        // Retourner une réponse JSON indiquant le succès de la modification
        $responseData = ['message' => 'Personne modifiée avec succès'];
        return new JsonResponse($responseData, JsonResponse::HTTP_OK);
    }

    /**
     * Supprimer une personne.
     *
     * API pour supprimer une personne.
     *
     */
    #[Route('/api/personne/{id}', name: 'api_personne_delete', methods: ['DELETE'])]
    #[OA\Response(
        response: 200,
        description: 'Returns the rewards of an user'
    )]
    #[OA\Parameter(
        name: 'id',
        in: 'query',
        description: 'The field used to order rewards',
        schema: new OA\Schema(type: 'int')
    )]


    public function delete(int $id, ManagerRegistry $doctrine): JsonResponse
    {
        // Récupérer la personne à supprimer depuis la base de données
        $entityManager = $doctrine->getManager();
        $personne = $entityManager->getRepository(Personne::class)->find($id);

        // Si la personne n'existe pas, retourner une erreur 404
        if (!$personne) {
            return new JsonResponse(['error' => 'Personne non trouvée'], JsonResponse::HTTP_NOT_FOUND);
        }

        // Supprimer le profil de la personne s'il existe
        $profile = $personne->getProfile();
        if ($profile) {
            $entityManager->remove($profile);
        }


        // Supprimer la personne de la base de données
        $entityManager->remove($personne);
        $entityManager->flush();

        // Retourner une réponse JSON indiquant le succès de la suppression
        return new JsonResponse(['message' => 'Personne supprimée avec succès'], JsonResponse::HTTP_OK);
    }


    #[Route('/api/personnes', name: 'api_personnes_search', methods: ['GET'])]
    public function search(Request $request, ManagerRegistry $doctrine): JsonResponse
    {
        // Récupérer les critères de recherche depuis la requête GET
        $criteria = $request->query->all();
//dd($criteria);
        // Construire la requête de recherche en fonction des critères
        $entityManager = $doctrine->getManager();
        $queryBuilder = $entityManager->createQueryBuilder();

        $queryBuilder->select('p.name, p.firstName,p.age, p.image, j.designation AS job_designation , pro.rs AS profile_rs, h.designation AS hobby_name')
            ->from(Personne::class, 'p')
            ->leftJoin('p.job', 'j')
            ->leftJoin('p.profile', 'pro')
            ->leftJoin('p.hobbies', 'h');

        // Ajouter les critères de recherche
        foreach ($criteria as $key => $value) {
            switch ($key) {
                case 'name':
                    $queryBuilder->andWhere('p.name LIKE :name')->setParameter('name', '%' . $value . '%');
                    break;
                case 'firstName':
                    $queryBuilder->andWhere('p.firstName LIKE :firstName')->setParameter('firstName', '%' . $value . '%');
                    break;
                case 'jobDescription':
                    $queryBuilder->andWhere('j.description LIKE :jobDescription')->setParameter('jobDescription', '%' . $value . '%');
                    break;
                case 'profileRs':
                    $queryBuilder->andWhere('pro.rs LIKE :profileRs')->setParameter('profileRs', '%' . $value . '%');
                    break;
                case 'age':
                    $queryBuilder->andWhere('p.age = :age')->setParameter('age', $value);
                    break;
                case 'hobbyDescription':
                    $queryBuilder->andWhere('h.description LIKE :hobbyDescription')->setParameter('hobbyDescription', '%' . $value . '%');
                    break;
                // Ajouter d'autres critères selon vos besoins
            }
        }

        $query = $queryBuilder->getQuery();
        $personnes = $query->getResult();

        // Retourner une réponse JSON avec les résultats de la recherche
        return new JsonResponse($personnes);
    }


    #[Route('/api/register', name: 'api_register', methods: ['POST'])]
    public function register(ManagerRegistry $doctrine, Request $request, UserPasswordHasherInterface $passwordEncoder, ValidatorInterface $validator): JsonResponse
    {
        // Récupérer les données envoyées via la méthode POST
        $data = json_decode($request->getContent(), true);

        // Vérifier si le nom d'utilisateur existe déjà
        $existingUser = $doctrine->getRepository(User::class)->findOneBy(['username' => $data['username']]);
        if ($existingUser) {
            return new JsonResponse(['message' => 'Le nom d\'utilisateur existe déjà'], JsonResponse::HTTP_BAD_REQUEST);
        }

        // Vérifier si l'email est valide
        if (!filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            return new JsonResponse(['message' => 'L\'adresse email n\'est pas valide'], JsonResponse::HTTP_BAD_REQUEST);
        }

        // Vérifier si le mot de passe a au moins 8 caractères
        if (strlen($data['password']) < 8) {
            return new JsonResponse(['message' => 'Le mot de passe doit avoir au moins 8 caractères'], JsonResponse::HTTP_BAD_REQUEST);
        }

        // Créer une nouvelle instance de l'entité User
        $user = new User();
        $user->setUsername($data['username']);
        $user->setEmail($data['email']);
        $user->setPassword(
            $passwordEncoder->hashPassword(
                $user,
                $data['password']
            )
        );

        // Valider l'utilisateur
        $errors = $validator->validate($user);
        if (count($errors) > 0) {
            return new JsonResponse(['message' => (string) $errors], JsonResponse::HTTP_BAD_REQUEST);
        }

        // Enregistrer l'utilisateur dans la base de données
        $entityManager = $doctrine->getManager();
        $entityManager->persist($user);
        $entityManager->flush();

        // Retourner une réponse JSON indiquant le succès de l'inscription
        return new JsonResponse(['message' => 'Inscription réussie'], JsonResponse::HTTP_CREATED);
    }


    #[Route('/api/login', name: 'api_login', methods: ['POST'])]
    public function login(TokenStorageInterface $tokenStorage, ManagerRegistry $doctrine, Request $request, UserPasswordHasherInterface $passwordEncoder): JsonResponse
    {
        // Récupérer les informations d'identification soumises
        $data = json_decode($request->getContent(), true);
        $email = $data['email'];
        $password = $data['password'];

        // Rechercher l'utilisateur dans la base de données par son email
        $user = $doctrine->getRepository(User::class)->findOneBy(['email' => $email]);

        // Vérifier si l'utilisateur existe
        if (!$user) {
            throw new BadCredentialsException('Email ou mot de passe incorrect');
        }

        // Vérifier si le mot de passe est correct
        if (!$passwordEncoder->isPasswordValid($user, $password)) {
            throw new BadCredentialsException('Email ou mot de passe incorrect');
        }

        // Création de la session utilisateur
        $token = new UsernamePasswordToken($user, "main", [], $user->getRoles());
        $tokenStorage->setToken($token);

        $tokenObject =$token->getUser();


        // Retourner le token dans la réponse JSON
        return new JsonResponse($tokenObject);
    }







}
